import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import requests
from PIL import Image, ImageTk
import io
import os

BASE = "http://192.168.0.10"  # IP der Kamera

THUMB_SIZE = 160
COLUMNS = 4

class OM1GUI:
    def __init__(self, root):
        self.root = root
        root.title("OM-1 Mark II Bilder laden - Du musst im WLAN der Kamera sein")
        root.geometry("1100x700")

        self.frame = ttk.Frame(root, padding=5)
        self.frame.pack(fill="both", expand=True)

        # ---------- TOP BAR ----------
        top = ttk.Frame(self.frame)
        top.pack(fill="x")

        self.slot_var = tk.IntVar(value=1)
        ttk.Radiobutton(top, text="Slot 1", variable=self.slot_var, value=1).pack(side="left")
        ttk.Radiobutton(top, text="Slot 2", variable=self.slot_var, value=2).pack(side="left")
        ttk.Button(top, text="Verbinden", command=self.connect).pack(side="left", padx=10)

        self.status = ttk.Label(top, text="Nicht verbunden")
        self.status.pack(side="left", padx=20)

        # ---------- MAIN SPLIT ----------
        main = ttk.PanedWindow(self.frame, orient=tk.HORIZONTAL)
        main.pack(fill="both", expand=True)

        # ---------- LEFT: FOLDERS ----------
        left = ttk.Frame(main, width=200)
        main.add(left, weight=1)

        ttk.Label(left, text="Ordner").pack(anchor="w")

        self.folder_list = tk.Listbox(left)
        self.folder_list.pack(fill="both", expand=True)
        self.folder_list.bind("<<ListboxSelect>>", self.load_images)

        # ---------- RIGHT: GALLERY ----------
        right = ttk.Frame(main)
        main.add(right, weight=4)

        self.canvas = tk.Canvas(right)
        self.scrollbar = ttk.Scrollbar(right, orient="vertical", command=self.canvas.yview)
        self.scroll_frame = ttk.Frame(self.canvas)

        self.scroll_frame.bind(
            "<Configure>",
            lambda e: self.canvas.configure(scrollregion=self.canvas.bbox("all"))
        )

        self.canvas.create_window((0, 0), window=self.scroll_frame, anchor="nw")
        self.canvas.configure(yscrollcommand=self.scrollbar.set)

        self.canvas.pack(side="left", fill="both", expand=True)
        self.scrollbar.pack(side="right", fill="y")

        # ---------- BOTTOM ----------
        bottom = ttk.Frame(self.frame)
        bottom.pack(fill="x")

        ttk.Button(bottom, text="Download markiert", command=self.download_selected).pack(side="left", padx=5)
        ttk.Button(bottom, text="Download alle", command=self.download_all).pack(side="left", padx=5)

        # ---------- STATE ----------
        self.thumbnail_cache = {}
        self.image_widgets = []
        self.selected = set()
        self.current_folder = None
        self.image_groups = {}

    # ---------- API ----------
    def api_get(self, cmd, params=None):
        url = f"{BASE}/{cmd}.cgi"
        r = requests.get(url, params=params, timeout=5)
        r.raise_for_status()
        return r.text.strip()

    def api_get_binary(self, cmd, params=None):
        url = f"{BASE}/{cmd}.cgi"
        r = requests.get(url, params=params, timeout=10)
        r.raise_for_status()
        return r.content

    # ---------- CONNECT ----------
    def connect(self):
        try:
            slot = self.slot_var.get()
            self.api_get("set_playtargetslot", {"targetslot": slot})
            self.status.config(text=f"Verbunden mit Slot {slot}")

            txt = self.api_get("get_imglist", {"DIR": "/DCIM"})
            lines = txt.splitlines()
            entries = lines[1].split(",")

            folders = [f for f in entries if f.startswith("1")]

            self.folder_list.delete(0, tk.END)
            for f in folders:
                self.folder_list.insert(tk.END, f)

            messagebox.showinfo("OK", f"{len(folders)} Ordner gefunden.")

        except Exception as e:
            messagebox.showerror("Fehler", str(e))

    # ---------- LOAD IMAGES ----------
    def clear_gallery(self):
        for w in self.image_widgets:
            w.destroy()
        self.image_widgets.clear()
        self.selected.clear()
        self.thumbnail_cache.clear()

    def load_images(self, event):
        if not self.folder_list.curselection():
            return

        folder = self.folder_list.get(self.folder_list.curselection()[0])
        self.current_folder = folder

        self.clear_gallery()

        txt = self.api_get("get_imglist", {"DIR": f"/DCIM/{folder}"})
        all_files = [f for f in txt.split(",") if "." in f]

        # Gruppieren nach Basisnamen
        groups = {}
        for f in all_files:
            base, ext = os.path.splitext(f)
            ext = ext.lower()
            if base not in groups:
                groups[base] = {}
            groups[base][ext] = f

        self.image_groups = groups

        display_items = []
        for base, exts in groups.items():
            if ".jpg" in exts:
                display_items.append((base, exts[".jpg"], exts.get(".orf")))
            elif ".orf" in exts:
                display_items.append((base, None, exts[".orf"]))

        # Galerie aufbauen
        row = 0
        col = 0

        for base, jpg, orf in display_items:
            if jpg:
                thumb = self.get_thumbnail(folder, jpg)
            else:
                thumb = self.make_placeholder_thumbnail("RAW")

            frame = tk.Frame(self.scroll_frame, bd=2, relief="flat")
            label = tk.Label(frame, image=thumb)
            label.image = thumb
            label.pack()
            cap = tk.Label(frame, text=base, wraplength=THUMB_SIZE)
            cap.pack()

            frame.grid(row=row, column=col, padx=5, pady=5)

            frame.bind("<Button-1>", lambda e, b=base, fr=frame: self.toggle_select(b, fr))
            label.bind("<Button-1>", lambda e, b=base, fr=frame: self.toggle_select(b, fr))

            self.image_widgets.append(frame)

            col += 1
            if col >= COLUMNS:
                col = 0
                row += 1

    # ---------- THUMBNAILS ----------
    def get_thumbnail(self, folder, filename):
        key = f"{folder}/{filename}"
        if key in self.thumbnail_cache:
            return self.thumbnail_cache[key]

        try:
            data = self.api_get_binary("get_thumbnail", {"DIR": f"/DCIM/{folder}/{filename}"})
            img = Image.open(io.BytesIO(data))
            img.thumbnail((THUMB_SIZE, THUMB_SIZE))
            tkimg = ImageTk.PhotoImage(img)
        except Exception:
            tkimg = self.make_placeholder_thumbnail("ERR")

        self.thumbnail_cache[key] = tkimg
        return tkimg

    def make_placeholder_thumbnail(self, text):
        img = Image.new("RGB", (THUMB_SIZE, THUMB_SIZE), (80, 80, 80))
        return ImageTk.PhotoImage(img)

    # ---------- SELECTION ----------
    def toggle_select(self, base, frame):
        if base in self.selected:
            self.selected.remove(base)
            frame.config(relief="flat", bg="SystemButtonFace")
        else:
            self.selected.add(base)
            frame.config(relief="solid", bg="#88ccff")

    # ---------- DOWNLOAD ----------
    def download_file(self, folder, filename, target_dir):
        url = f"{BASE}/DCIM/{folder}/{filename}"
        r = requests.get(url, stream=True)
        r.raise_for_status()

        path = os.path.join(target_dir, filename)
        with open(path, "wb") as f:
            for chunk in r.iter_content(8192):
                f.write(chunk)

    def download_selected(self):
        if not self.selected:
            messagebox.showwarning("Hinweis", "Keine Bilder ausgewählt.")
            return

        target = filedialog.askdirectory()
        if not target:
            return

        for base in self.selected:
            exts = self.image_groups.get(base, {})

            # Immer ORF zuerst
            if ".orf" in exts:
                fname = exts[".orf"]
                self.status.config(text=f"Lade {fname} ...")
                self.root.update()
                self.download_file(self.current_folder, fname, target)

            # Dann JPG wenn vorhanden
            if ".jpg" in exts:
                fname = exts[".jpg"]
                self.status.config(text=f"Lade {fname} ...")
                self.root.update()
                self.download_file(self.current_folder, fname, target)

        self.status.config(text="Download fertig.")

    def download_all(self):
        target = filedialog.askdirectory()
        if not target:
            return

        for base, exts in self.image_groups.items():
            if ".orf" in exts:
                self.download_file(self.current_folder, exts[".orf"], target)
            if ".jpg" in exts:
                self.download_file(self.current_folder, exts[".jpg"], target)

        self.status.config(text="Alle Downloads fertig.")

# ---------- MAIN ----------
if __name__ == "__main__":
    root = tk.Tk()
    app = OM1GUI(root)
    root.mainloop()
