import html
import xml.dom.minidom as minidom
from pathlib import Path
import tkinter as tk
from tkinter import filedialog, messagebox

TAG_COLOR = "#005cc5"
ATTR_NAME_COLOR = "#6f42c1"
ATTR_VALUE_COLOR = "#032f62"
TEXT_COLOR = "#24292e"
INDENT_PER_LEVEL = 25
DEFAULT_COLLAPSE_LEVEL = 2

class LineCounter:
    def __init__(self):
        self.value = 1
    def next(self):
        v = self.value
        self.value += 1
        return v

def format_open_tag(node):
    parts = [f'<span style="color:{TAG_COLOR}">&lt;</span>']
    parts.append(f'<span style="color:{TAG_COLOR}">{html.escape(node.nodeName)}</span>')
    if node.attributes:
        for i in range(node.attributes.length):
            attr = node.attributes.item(i)
            parts.append(" ")
            parts.append(f'<span style="color:{ATTR_NAME_COLOR}">{html.escape(attr.name)}=</span>')
            parts.append(f'<span style="color:{ATTR_VALUE_COLOR}">&quot;{html.escape(attr.value)}&quot;</span>')
    parts.append(f'<span style="color:{TAG_COLOR}">&gt;</span>')
    return "".join(parts)

def format_close_tag(node):
    return (f'<span style="color:{TAG_COLOR}">&lt;/</span>'
            f'<span style="color:{TAG_COLOR}">{html.escape(node.nodeName)}</span>'
            f'<span style="color:{TAG_COLOR}">&gt;</span>')

def render_node(node, level, counter, max_level=None):
    indent = level * INDENT_PER_LEVEL
    if node.nodeType != node.ELEMENT_NODE:
        return ""

    ln = counter.next()
    open_tag = format_open_tag(node)
    close_tag = format_close_tag(node)

    # Prüfen, ob Text-only Node (keine Element-Kinder)
    text_only = all(c.nodeType != c.ELEMENT_NODE for c in node.childNodes)

    if text_only:
        # Alles in einer Zeile rendern
        text_content = "".join(
            html.escape(c.data) if c.nodeType == c.TEXT_NODE else ""
            for c in node.childNodes
        )
        one_line = f"{open_tag}<span style='color:{TEXT_COLOR}'>{text_content}</span>{close_tag}"
        return f'<div class="node" data-level="{level}" data-textonly="true">' \
               f'<div class="row">' \
               f'<div class="lineno">{ln}</div>' \
               f'<div class="code" style="padding-left:{indent}px;">' \
               f'<span class="arrow"></span>{one_line}</div></div></div>'

    # Nodes mit Kinder-Elementen bleiben einklappbar
    short_tag = f"{open_tag}<span style='color:#999'>...</span>{close_tag}"
    collapsed = max_level is not None and level >= max_level
    arrow = "▶" if collapsed else "▼"
    arrow_color = "#666" if collapsed else "orange"

    html_out = f'<div class="node" data-level="{level}" data-textonly="false">'
    html_out += f'<div class="row">' \
                f'<div class="lineno">{ln}</div>' \
                f'<div class="code" style="padding-left:{indent}px;" ' \
                f'data-open="{html.escape(open_tag)}" data-short="{html.escape(short_tag)}" onclick="toggleNode(event)">' \
                f'<span class="arrow" style="color:{arrow_color}">{arrow}</span>' \
                f'<span class="tagcontent">{short_tag if collapsed else open_tag}</span>' \
                f'</div></div>'

    html_out += f'<div class="children" style="display:{"none" if collapsed else "block"};">'
    for child in node.childNodes:
        if child.nodeType == node.ELEMENT_NODE:
            html_out += render_node(child, level+1, counter, max_level)
        elif child.nodeType == node.TEXT_NODE:
            text = child.data
            if text.strip() != "":
                ln = counter.next()
                html_out += f'<div class="row"><div class="lineno">{ln}</div>' \
                            f'<div class="code" style="padding-left:{(level+1)*INDENT_PER_LEVEL}px;">' \
                            f'<span class="arrow"></span>' \
                            f'<span style="color:{TEXT_COLOR}">{html.escape(text)}</span></div></div>'
        elif child.nodeType == child.CDATA_SECTION_NODE:
            ln = counter.next()
            html_out += f'<div class="row"><div class="lineno">{ln}</div>' \
                        f'<div class="code" style="padding-left:{(level+1)*INDENT_PER_LEVEL}px;">' \
                        f'<span class="arrow"></span>' \
                        f'<span style="color:{TEXT_COLOR}">&lt;![CDATA[{html.escape(child.data)}]]&gt;</span></div></div>'
        elif child.nodeType == child.COMMENT_NODE:
            ln = counter.next()
            html_out += f'<div class="row"><div class="lineno">{ln}</div>' \
                        f'<div class="code" style="padding-left:{(level+1)*INDENT_PER_LEVEL}px;">' \
                        f'<span class="arrow"></span>' \
                        f'<span style="color:{TEXT_COLOR}">&lt;!-- {html.escape(child.data)} --&gt;</span></div></div>'

    # Schließ-Tag für Nodes mit Kinder-Elementen
    ln = counter.next()
    html_out += f'<div class="row"><div class="lineno">{ln}</div>' \
                f'<div class="code" style="padding-left:{indent}px;"><span class="arrow"></span>{close_tag}</div></div>'
    html_out += '</div></div>'
    return html_out

def render_html_document_from_xml(xml_string, title="XML Viewer"):
    dom = minidom.parseString(xml_string)
    root = dom.documentElement
    counter = LineCounter()
    body = render_node(root, 0, counter, max_level=DEFAULT_COLLAPSE_LEVEL)
    return f"""<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="UTF-8">
<title>{html.escape(title)}</title>
<style>
body {{ font-family: Consolas, monospace; font-size: 14px; margin: 20px; }}
.row {{ display: flex; align-items: flex-start; line-height:1.25; }}
.lineno {{ width: 60px; text-align:right; padding-right:10px; color:#999; user-select:none; flex-shrink:0; }}
.code {{ display:inline-block; white-space:pre-wrap; word-break:break-word; cursor:pointer; flex:1; }}
.arrow {{ display:inline-block; width:16px; user-select:none; }}
#controls {{ margin-bottom:10px; }}
#sliderContainer {{ display:inline-flex; align-items:center; margin-top:5px; }}
#levelSlider {{ max-width:5cm; margin-left:5px; margin-right:5px; }}
</style>
</head>
<body>
<h1>{html.escape(title)}</h1>

<div id="controls">
<button onclick="expandAll()">Alle aufklappen</button>
<button onclick="collapseAll()">Alle zuklappen</button>
<br>
<label for="levelSlider">Maximal einklappen bis Ebene:</label>
<span id="sliderContainer">
<input type="range" id="levelSlider" min="0" max="10" value="{DEFAULT_COLLAPSE_LEVEL}" oninput="applySlider(this.value)">
<span id="sliderValue">{DEFAULT_COLLAPSE_LEVEL}</span>
</span>
</div>

{body}

<script>
function toggleNode(event){{
    const codeDiv = event.currentTarget;
    const node = codeDiv.closest(".node");
    const children = node.querySelector(".children");
    if(!children) return;

    const arrow = codeDiv.querySelector(".arrow");
    const tagSpan = codeDiv.querySelector(".tagcontent");
    const openHtml = codeDiv.dataset.open;
    const shortHtml = codeDiv.dataset.short;

    if(children.style.display==="none") {{
        children.style.display="block";
        arrow.textContent="▼";
        arrow.style.color="orange";
        tagSpan.innerHTML=openHtml;
    }} else {{
        children.style.display="none";
        arrow.textContent="▶";
        arrow.style.color="#666";
        tagSpan.innerHTML=shortHtml;
    }}
}}

function expandAll(){{
    document.querySelectorAll(".children").forEach(c=>c.style.display="block");
    document.querySelectorAll(".arrow").forEach(a=>{{a.textContent="▼"; a.style.color="orange";}});
    document.querySelectorAll(".code").forEach(c=>{{let t=c.querySelector(".tagcontent"); if(t&&c.dataset.open)t.innerHTML=c.dataset.open;}});
}}

function collapseAll(){{
    document.querySelectorAll(".children").forEach(c=>c.style.display="none");
    document.querySelectorAll(".arrow").forEach(a=>{{a.textContent="▶"; a.style.color="#666";}});
    document.querySelectorAll(".code").forEach(c=>{{let t=c.querySelector(".tagcontent"); if(t&&c.dataset.short)t.innerHTML=c.dataset.short;}});
}}

function applySlider(level){{
    document.getElementById("sliderValue").textContent = level;
    document.querySelectorAll(".node").forEach(n=>{{
        const nodeLevel = parseInt(n.dataset.level);
        const children = n.querySelector(".children");
        const codeDiv = n.querySelector(".code");
        const arrow = codeDiv.querySelector(".arrow");
        const tagSpan = codeDiv.querySelector(".tagcontent");
        if(!children) return;
        if(nodeLevel >= level){{
            children.style.display="none";
            arrow.textContent="▶";
            arrow.style.color="#666";
            tagSpan.innerHTML=codeDiv.dataset.short;
        }} else {{
            children.style.display="block";
            arrow.textContent="▼";
            arrow.style.color="orange";
            tagSpan.innerHTML=codeDiv.dataset.open;
        }}
    }});
}}
applySlider({DEFAULT_COLLAPSE_LEVEL});
</script>

</body>
</html>
"""

def main():
    root = tk.Tk()
    root.title("XML zu HTML Converter")
    root.withdraw()

    in_file = filedialog.askopenfilename(
        title="XML-Datei auswählen",
        filetypes=[("XML Dateien","*.xml"),("Alle Dateien","*.*")]
    )
    if not in_file:
        return

    out_file = filedialog.asksaveasfilename(
        title="HTML-Datei speichern",
        defaultextension=".html",
        filetypes=[("HTML Dateien","*.html"),("Alle Dateien","*.*")]
    )
    if not out_file:
        return

    try:
        xml_text = Path(in_file).read_text(encoding="utf-8")
        html_doc = render_html_document_from_xml(xml_text, title=Path(in_file).name)
        Path(out_file).write_text(html_doc, encoding="utf-8")
        messagebox.showinfo("Fertig", f"HTML wurde erzeugt:\n{out_file}")
    except Exception as e:
        messagebox.showerror("Fehler", str(e))

if __name__=="__main__":
    main()
