#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import time
from itertools import product
import requests

# =============================================
# Konfiguration
# =============================================
CAM_IP = "192.168.0.10"   # ggf. anpassen
LIVEVIEW_PORT = 5555
REQUEST_TIMEOUT = 6
DELAY_AFTER_SET = 0.15
DELAY_AFTER_SHOT = 0.25

# =============================================
# Standardreihen (1/3 EV)
# =============================================

SHUTTER_1_3 = [
    1/8000, 1/6400, 1/5000,
    1/4000, 1/3200, 1/2500,
    1/2000, 1/1600, 1/1250,
    1/1000, 1/800,  1/640,
    1/500,  1/400,  1/320,
    1/250,  1/200,  1/160,
    1/125,  1/100,  1/80,
    1/60,   1/50,   1/40,
    1/30,   1/25,   1/20,
    1/15,   1/13,   1/10,
    1/8,    1/6,    1/5,
    1/4,    1/3,    0.4,
    0.5,    0.6,    0.8,
    1.0,    1.3,    1.6,
    2.0,    2.5,    3.2,
    4.0,    5.0,    6.0,
    8.0,    10.0,   13.0,
    15.0,   20.0,   25.0,
    30.0
]

APERTURE_1_3 = [
    1.0, 1.1, 1.2, 1.4, 1.6, 1.8,
    2.0, 2.2, 2.5, 2.8, 3.2, 3.5,
    4.0, 4.5, 5.0, 5.6, 6.3, 7.1,
    8.0, 9.0, 10.0, 11.0, 13.0, 14.0,
    16.0, 18.0, 20.0, 22.0
]

ISO_1_3 = [
    100, 125, 160,
    200, 250, 320,
    400, 500, 640,
    800, 1000, 1250,
    1600, 2000, 2500,
    3200, 4000, 5000,
    6400, 8000, 10000,
    12800, 16000, 20000,
    25600, 32000, 40000,
    51200
]

# =============================================
# Eingabe- und Formatfunktionen
# =============================================
def parse_shutter_value(s: str) -> float:
    s = s.strip().lower()
    if "/" in s:
        a, b = s.split("/", 1)
        return float(a) / float(b)
    return float(s)

def format_shutter_for_camera(t: float) -> str:
    inv = 1.0 / t
    if abs(inv - round(inv)) < 1e-9 and inv >= 1:
        return str(int(round(inv)))
    if abs(t - round(t)) < 1e-9:
        return str(int(round(t)))
    return f"{t:.4f}".rstrip("0").rstrip(".")

def parse_aperture_input(s: str) -> float:
    s = s.strip().lower().replace(" ", "")
    if s.startswith("f/"):
        s = s[2:]
    elif s.startswith("f"):
        s = s[1:]
    return float(s)

def parse_iso_input(s: str) -> int:
    s = s.strip().lower().replace(" ", "").replace("iso", "")
    return int(float(s))

def parse_ev_step(s: str) -> str:
    s = s.strip().lower().replace(" ", "")
    if s in {"1/3ev", "1/3"}: return "1/3"
    if s in {"1/2ev", "1/2"}: return "1/2"
    if s in {"1ev",  "1"}:   return "1"
    raise ValueError("EV-Schrittweite muss 1/3, 1/2 oder 1 sein.")

# =============================================
# Rasterfunktionen
# =============================================
def thirds_to_step(lst, step):
    if step == "1/3": return list(lst)
    if step == "1/2": return [v for i,v in enumerate(lst) if i % 2 == 0]
    if step == "1":   return [v for i,v in enumerate(lst) if i % 3 == 0]
    raise ValueError("Unbekannter EV-Schritt.")

def snap_range(values, start, end):
    lo, hi = (start, end) if start <= end else (end, start)
    cut = [v for v in values if lo - 1e-12 <= v <= hi + 1e-12]
    return cut if start <= end else list(reversed(cut))

# =============================================
# HTTP-Funktionen
# =============================================
def http_get(session, path, params=None):
    url = f"http://{CAM_IP}/{path}"
    r = session.get(url, params=params, timeout=REQUEST_TIMEOUT)
    r.raise_for_status()
    return r

def http_post_value(session, propname, value):
    url = f"http://{CAM_IP}/set_camprop.cgi"
    body = f"<set><value>{value}</value></set>"
    r = session.post(url,
                     params={"com": "set", "propname": propname},
                     data=body,
                     headers={"Content-Type":"application/x-www-form-urlencoded"},
                     timeout=REQUEST_TIMEOUT)
    r.raise_for_status()
    return r

def switch_to_rec(session):
    return http_get(session, "switch_cammode.cgi", {"mode": "rec"})

def start_liveview(session):
    return http_get(session, "exec_takemisc.cgi",
                    {"com": "startliveview", "port": str(LIVEVIEW_PORT)})

def stop_liveview(session):
    return http_get(session, "exec_takemisc.cgi", {"com": "stopliveview"})

def trigger(session):
    return http_get(session, "exec_takemotion.cgi", {"com": "starttake"})

# =============================================
# NEU: Verbindungscheck
# =============================================
def connection_check():
    try:
        url = f"http://{CAM_IP}/get_caminfo"
        r = requests.get(url, timeout=REQUEST_TIMEOUT)
        if r.status_code == 200:
            print("Kamera erreichbar.")
            return True
        print("Antwort erhalten, aber Status != 200.")
        return False
    except Exception:
        print("Kamera nicht erreichbar.")
        return False

# =============================================
# Hauptprogramm
# =============================================
def main():

    print("\nEV‑Bracketing für OM1.2")

    print("\nVerbindungscheck (ist die Kamera erreichbar?)…")
    ok = connection_check()
    if not ok:
        q = input("Trotzdem fortfahren? (j/n): ").strip().lower()
        if q not in {"j", "ja", "y"}:
            print("Abgebrochen.")
            return

    # Eingaben
    print("\nGib die Start- und Endwerte der Belichtungsparameter an und die Schrittweite:\n")
    sh_start = parse_shutter_value(input("Belichtungszeit Startwert (z.B. 1/100 oder 1): ").strip())
    sh_end   = parse_shutter_value(input("Belichtungszeit Endwert (z.B. 1/30 oder 2): ").strip())
    f_start  = parse_aperture_input(input("Blende Startwert (z.B. 2.8): ").strip())
    f_end    = parse_aperture_input(input("Blende Endwert (z.B. 4): ").strip())
    iso_start= parse_iso_input(input("ISO Startwert (z.B. 200): ").strip())
    iso_end  = parse_iso_input(input("ISO Endwert (z.B. 400): ").strip())
    ev_step  = parse_ev_step(input("EV-Schrittweite (1/3, 1/2, 1): "))

    # Raster
    shutter_grid  = thirds_to_step(SHUTTER_1_3, ev_step)
    aperture_grid = thirds_to_step(APERTURE_1_3, ev_step)
    iso_grid      = thirds_to_step(ISO_1_3, ev_step)

    shutter_series  = snap_range(shutter_grid, sh_start, sh_end)
    aperture_series = snap_range(aperture_grid, f_start, f_end)
    iso_series      = [int(v) for v in snap_range(iso_grid, iso_start, iso_end)]

    shutter_fmt = [format_shutter_for_camera(v) for v in shutter_series]
    aperture_fmt = [str(v).rstrip("0").rstrip(".") for v in aperture_series]

    total = len(shutter_series)*len(aperture_series)*len(iso_series)

    print("\nSerien:")
    print("  Zeit   :", shutter_fmt)
    print("  Blende :", aperture_fmt)
    print("  ISO    :", iso_series)
    print("→ Gesamt:", total)

    if input("Fortfahren (j/n)? ").strip().lower() not in {"j","ja","y"}:
        print("Abgebrochen.")
        return

    # Aufnahmeprozess
    with requests.Session() as session:
        try: switch_to_rec(session)
        except Exception as e: print("REC-Modus warn:", e)

        try: start_liveview(session)
        except Exception as e: print("LiveView warn:", e)

        idx = 0
        for t, fval, iso in product(shutter_series, aperture_series, iso_series):
            idx += 1
            sh = format_shutter_for_camera(t)
            f  = str(fval).rstrip("0").rstrip(".")
            i  = str(iso)

            print(f"[{idx}/{total}] T={sh} f={f} ISO={i}")

            try:
                http_post_value(session, "focalvalue",     f)
                time.sleep(DELAY_AFTER_SET)
                http_post_value(session, "isospeedvalue",  i)
                time.sleep(DELAY_AFTER_SET)
                http_post_value(session, "shutspeedvalue", sh)
                time.sleep(DELAY_AFTER_SET)
                trigger(session)
                time.sleep(DELAY_AFTER_SHOT)
            except Exception as e:
                print("Fehler:", e)

        try: stop_liveview(session)
        except Exception: pass

    print("Fertig.")

if __name__ == "__main__":
    main()