import requests
import xml.etree.ElementTree as ET
import time

CAMERA_IP = "192.168.0.10"
BASE_URL = f"http://{CAMERA_IP}"
MODE_SWITCH_WAIT = 2  # Sekunden warten nach Moduswechsel

# Mapping mit Reihenfolge der Ausgabe
PROP_NAME_DE_ORDERED = [
    ("shutspeedvalue", "Belichtungszeit"),
    ("isospeedvalue", "ISO-Empfindlichkeit"),
    ("focalvalue", "Blende / Brennweite"),
    ("expcomp", "Belichtungskorrektur"),
    ("wbvalue", "Weißabgleich"),
    ("drivemode", "Drive-Modus"),
    ("takemode", "Aufnahmemodus"),
    ("touchactiveframe", "Touch-Aktivbereich"),
    ("noisereduction", "Rauschreduzierung"),
    ("lowvibtime", "Low-Vibration Zeit"),
    ("bulbtimelimit", "Bulb Limit"),
    ("digitaltelecon", "Digital-Telekonverter"),
    ("supermacrozoom", "Super-Makro Zoom"),
    ("exposemovie", "Movie Belichtung"),
    ("cameradrivemode", "Kamera Modus"),
    ("SilentNoiseReduction", "Silent Rauschreduzierung"),
    ("SilentTime", "Silent Zeit"),
    ("QualityMovie2", "Movie Qualität"),
    ("NoiseReductionExposureTime", "Rauschreduktion Belichtungszeit")
]

# Breite der Spalten für sauberen Output
COL1_WIDTH = 28  # Property-Name
COL2_WIDTH = 32  # Deutsche Bezeichnung

def set_rec_mode():
    url = f"{BASE_URL}/switch_cammode.cgi"
    params = {"mode": "rec"}
    try:
        r = requests.get(url, params=params, timeout=5)
        if r.status_code == 200:
            time.sleep(MODE_SWITCH_WAIT)
            return True
        else:
            print(f"Fehler beim Moduswechsel: HTTP {r.status_code}")
            return False
    except Exception as e:
        print(f"FEHLER beim Moduswechsel: {e}")
        return False

def get_desclist():
    url = f"{BASE_URL}/get_camprop.cgi"
    params = {"com": "desc", "propname": "desclist"}
    try:
        r = requests.get(url, params=params, timeout=5)
        if r.status_code == 200:
            return r.text
        else:
            print(f"HTTP {r.status_code} beim Abrufen der Desclist")
            return ""
    except Exception as e:
        print(f"FEHLER bei Desclist: {e}")
        return ""

def parse_desclist(xml_text):
    props = {}
    try:
        root = ET.fromstring(xml_text)
        for desc in root.findall('desc'):
            propname = desc.find('propname').text if desc.find('propname') is not None else "UNKNOWN"
            value = desc.find('value').text if desc.find('value') is not None else "(kein Wert)"
            props[propname] = value
        return props
    except ET.ParseError as e:
        print(f"XML Parse Fehler: {e}")
        return {}

def main():
    print(f"Kamera: {CAMERA_IP}\n")
    
    if not set_rec_mode():
        print("Moduswechsel fehlgeschlagen. Beende Script.")
        return

    xml_text = get_desclist()
    if not xml_text:
        print("Keine Desclist erhalten. Kamera evtl. privat oder Fehler")
        return

    properties = parse_desclist(xml_text)
    if not properties:
        print("Keine Properties gefunden.")
        return

    # Ausgabe mit sauber ausgerichteten Spalten
    for propname, deutsch in PROP_NAME_DE_ORDERED:
        value = properties.get(propname, "(nicht verfügbar)")
        print(f"{propname.ljust(COL1_WIDTH)}{deutsch.ljust(COL2_WIDTH)}{value}")

if __name__ == "__main__":
    main()
