import requests
import time
import os
import csv
from collections import defaultdict

CAMERA_IP = "192.168.0.10"
BASE_URL = f"http://{CAMERA_IP}"

OUTPUT_DIR = "camera_sets"
os.makedirs(OUTPUT_DIR, exist_ok=True)

SESSION = requests.Session()
SESSION.timeout = 20


def cam_get(path, params=None):
    url = BASE_URL + path
    r = SESSION.get(url, params=params)
    r.raise_for_status()
    return r


def check_connection():
    cam_get("/get_caminfo.cgi")
    print("Camera reachable")


def enter_maintenance():
    cam_get("/switch_cammode.cgi", params={"mode": "maintenance"})
    print("Maintenance mode activated")
    time.sleep(1)


def request_snapshot(mode, kind):
    r = cam_get("/request_getmysetdata.cgi", params={
        "mode": mode,
        "kind": kind
    })
    return r.text


def get_size():
    r = cam_get("/get_mysetdatasize.cgi", params={"kind": "current"})
    text = r.text
    import re
    m = re.search(r"<datasize>(\d+)</datasize>", text)
    if not m:
        return None
    return int(m.group(1))


def download_blob(size):
    r = SESSION.get(
        BASE_URL + "/get_partialmysetdata.cgi",
        params={
            "kind": "current",
            "offset": 0,
            "size": size
        },
        stream=True
    )
    r.raise_for_status()
    return r.content


def probe_set(name, mode, kind):
    try:
        txt = request_snapshot(mode, kind)
        if "error" in txt.lower():
            return False
        time.sleep(1)
        size = get_size()
        if size is None or size == 0:
            return False
        print(f"{name} exists, size = {size}")
        return True
    except:
        return False


def download_set(name, mode, kind):
    request_snapshot(mode, kind)
    time.sleep(1.2)
    size = get_size()
    if not size:
        raise RuntimeError("Could not get datasize")
    data = download_blob(size)
    return data.decode(errors="ignore")


def parse_set_full(data):
    """
    Alle Zeilen der Dump-Datei aufnehmen.
    Key = Zeileninhalt (mit Index bei Duplikaten)
    Wert = gleiche Zeile
    """
    result = {}
    counts = defaultdict(int)
    for line in data.splitlines():
        line = line.strip()
        if not line:
            continue
        counts[line] += 1
        key = line if counts[line] == 1 else f"{line}_{counts[line]}"
        result[key] = line
    return result


def main():
    print("OM-1 Mark II MySet CSV Exporter")
    check_connection()
    enter_maintenance()

    # Kandidaten auf OM-1 Mark II beschränkt (4 Custom Sets)
    candidates = [
        ("current", "current", "current"),
        ("myset1", "myset1", "current"),
        ("myset2", "myset2", "current"),
        ("myset3", "myset3", "current"),
        ("myset4", "myset4", "current"),
    ]

    available = []
    print("Scanning for available sets...")
    for name, mode, kind in candidates:
        if probe_set(name, mode, kind):
            available.append((name, mode, kind))

    if not available:
        print("No sets found")
        return

    print("\nAvailable sets:")
    for i, (name, _, _) in enumerate(available, 1):
        print(f"{i}) {name}")

    choice = input("\nEnter the numbers of the sets to download (e.g., 1 2 4): ")
    try:
        indices = [int(x.strip()) - 1 for x in choice.strip().split()]
    except:
        print("Invalid input")
        return

    selected_sets = []
    for idx in indices:
        if 0 <= idx < len(available):
            name, mode, kind = available[idx]
            print(f"Downloading {name}...")
            data = download_set(name, mode, kind)
            parsed = parse_set_full(data)
            selected_sets.append((name, parsed))
        else:
            print(f"Skipping invalid index: {idx+1}")

    if not selected_sets:
        print("No sets selected")
        return

    # Alle Keys sammeln
    all_keys = set()
    for _, d in selected_sets:
        all_keys.update(d.keys())
    all_keys = sorted(all_keys)

    # CSV schreiben
    csv_file = os.path.join(OUTPUT_DIR, "mysets_comparison.csv")
    with open(csv_file, "w", newline="", encoding="utf-8") as f:
        writer = csv.writer(f)
        # Header
        header = ["Parameter"] + [name for name, _ in selected_sets]
        writer.writerow(header)
        # Zeilen
        for key in all_keys:
            row = [key]
            for _, d in selected_sets:
                row.append(d.get(key, ""))
            writer.writerow(row)

    print(f"\nDone. CSV saved to {csv_file}")


if __name__ == "__main__":
    main()
